<?php
declare(strict_types=1);

function tep_db_connect(
    string $server = DB_SERVER,
    string $username = DB_SERVER_USERNAME,
    string $password = DB_SERVER_PASSWORD,
    string $database = DB_DATABASE
): mysqli {
    if (USE_PCONNECT === 'true') {
        $server = 'p:' . $server;
    }

    $link = mysqli_connect($server, $username, $password, $database);

    if (!$link) {
        throw new RuntimeException('Connection failed: ' . mysqli_connect_error());
    }

    mysqli_set_charset($link, 'utf8');
    return $link;
}

function tep_db_close(mysqli $link): bool {
    return mysqli_close($link);
}

function tep_db_error(string $query, int $errno, string $error): void {
    global $logger;

    if (defined('STORE_DB_TRANSACTIONS') && STORE_DB_TRANSACTIONS === 'true') {
        $logger?->write("[" . date('Y-m-d H:i:s') . "] [$errno] $error", 'ERROR');
    }

    die("<strong>$errno - " . htmlspecialchars($error) . "<br><br>" . htmlspecialchars($query) . "<br><br><span style='color:red'>[TEP STOP]</span></strong>");
}

function tep_db_query(string $query, mysqli $link): mysqli_result|false {
    global $logger;

    if (defined('STORE_DB_TRANSACTIONS') && STORE_DB_TRANSACTIONS === 'true') {
        $logger ??= new logger();
        $logger->write($query, 'QUERY');
    }

    $result = mysqli_query($link, $query);
    if (!$result) {
        tep_db_error($query, mysqli_errno($link), mysqli_error($link));
    }

    return $result;
}

function tep_db_perform(string $table, array $data, string $action = 'insert', string $parameters = '', mysqli $link): mysqli_result|false {
    if ($action === 'insert') {
        $columns = implode(', ', array_keys($data));
        $values = array_map(function ($value) use ($link) {
            return match ((string)$value) {
                'now()' => 'now()',
                'null' => 'null',
                default => "'" . tep_db_input($value, $link) . "'",
            };
        }, array_values($data));

        $query = "INSERT INTO $table ($columns) VALUES (" . implode(', ', $values) . ")";
    } elseif ($action === 'update') {
        $updates = [];
        foreach ($data as $column => $value) {
            $updates[] = match ((string)$value) {
                'now()' => "$column = now()",
                'null' => "$column = null",
                default => "$column = '" . tep_db_input($value, $link) . "'",
            };
        }
        $query = "UPDATE $table SET " . implode(', ', $updates) . " WHERE $parameters";
    } else {
        throw new InvalidArgumentException("Unsupported action: $action");
    }

    return tep_db_query($query, $link);
}

// Utility wrappers
function tep_db_fetch_array(mysqli_result $db_query): array|null {
    return mysqli_fetch_array($db_query, MYSQLI_ASSOC);
}

function tep_db_result(mysqli_result $result, int $row, string|int $field = 0): mixed {
    tep_db_data_seek($result, $row);
    $data = tep_db_fetch_array($result);
    return $data[$field] ?? null;
}

function tep_db_num_rows(?mysqli_result $db_query): int {
    return $db_query ? mysqli_num_rows($db_query) : 0;
}

function tep_db_data_seek(mysqli_result $db_query, int $row_number): bool {
    return mysqli_data_seek($db_query, $row_number);
}

function tep_db_insert_id(mysqli $link): int|string {
    return mysqli_insert_id($link);
}

function tep_db_free_result(mysqli_result $db_query): bool {
    return mysqli_free_result($db_query);
}

function tep_db_fetch_fields(mysqli_result $db_query): object|false {
    return mysqli_fetch_field($db_query);
}

function tep_db_output(string $string): string {
    return htmlspecialchars($string, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function tep_db_input(string $string, mysqli $link): string {
    return mysqli_real_escape_string($link, $string);
}

function tep_db_prepare_input(mixed $input): mixed {
    return match (true) {
        is_string($input) => trim(stripslashes($input)),
        is_array($input) => array_map('tep_db_prepare_input', $input),
        default => $input,
    };
}

function tep_db_affected_rows(mysqli $link): int {
    return mysqli_affected_rows($link);
}

function tep_db_get_server_info(mysqli $link): string {
    return mysqli_get_server_info($link);
}